program MkBoot;

{$I-}
{$C-}

{reads three files to produce boot image file}

{
  First file is boot loader HEX file.
  Must be compiled to start at address 0.
  Boot loader must fit into first 256 bytes.

  Second file is CPM image HEX file.
  The start address of this file is used
  as the base load address of the CPM image.

  The third file is the BIOS HEX file.
  The start address of this file is used
  as the cold boot execution address of the
  combined system image (CPM+BIOS).
  The BIOS address range can over;ap the CPM image.

  The combined CPM + BIOS image size must
  have a combined byte size greater
  then (8192 - 256) bytes.
}

const
  C_BootRecBytes = 8192;
  C_MaxBootRecIndex = 8191;
  C_MaxLoaderBytes = 256;

type
  PChars = ^TChars;
  TChars = array[0..255] of char;

  TInputLine = string[255];

  TFileName = string[15];

  PHexLine = ^THexLine;
  THexLine = record
    DataCount: byte;
    Address: integer;
    RecType: byte;
    ChkSum: byte;
    Data: array[0..255] of byte;
  end;

  TBootRec = array[0..C_MaxBootRecIndex] of byte;


function CharToNibble(Ch: char): byte;

var
  aByte: byte;

begin
  if ((Ch >= '0') and (Ch <= '9')) then begin
    aByte:= ord(Ch) - Ord('0')
  end
  else begin
    if ((Ch >= 'A') and (Ch <= 'F')) then begin
      aByte:= ord(Ch) - Ord('A') + 10
    end
  end;
  CharToNibble:= aByte
end;


function HexToByte(HexChars: PChars): byte;

var
  Value: byte;
  Ch1, Ch2: char;

begin
  {MSB first}
  Ch1:= char(HexChars^[0]);
  Ch2:= char(HexChars^[1]);
  Value:= (CharToNibble(Ch1) shl 4) or CharToNibble(Ch2);
  HexToByte:= Value
end;


function HexToWord(HexChars: PChars): integer;


var
  B1: integer;
  Res: integer;

begin
  B1:= HexToByte(HexChars);
  HexChars:= Ptr(addr(HexChars^[2]));
  Res:= HexToByte(HexChars);
  Res:= (B1 shl 8) or Res;
  HexToWord:= Res
end;


function DecodeHexLine(var Line: TInputLine;
                       var Data: THexLine;
                       var ChkSum: byte): boolean;

var
  InLen, ReqLen, Temp: byte;
  CurChar: PChars;
  Idx: byte;
  WasOk: boolean;

begin
  WasOk:= false;
  InLen:= Length(Line);
  ChkSum:= 0;
  if (InLen > 10) and (Line[1] = ':') then begin
    WasOk:= true;
    for Idx:= 2 to InLen do begin
      if not (Line[Idx] in ['0'..'9', 'A'..'F']) then begin
        WasOk:= false
      end
    end;
    if WasOk then begin
      Data.ChkSum:= 0;
      CurChar:= Ptr(addr(Line[2]));
      Data.DataCount:= HexToByte(CurChar);
      CurChar:= Ptr(addr(Line[4]));
      Data.Address:= HexToWord(CurChar);
      CurChar:= Ptr(addr(Line[8]));
      Data.RecType:= HexToByte(CurChar);
      {initialize check sum}
      ChkSum:= ChkSum + Data.DataCount + Hi(Data.Address) + Lo(Data.Address);
      ChkSum:= ChkSum + Data.RecType;
      {make sure length of Line = 11 + (DataCount * 2)}
      ReqLen:= 11 + (Data.DataCount * 2);
      If (ReqLen = InLen) then begin
        CurChar:= Ptr(addr(Line[10]));
        if (Data.DataCount > 0) then begin
          for Idx:=0 to Data.DataCount - 1 do begin
            Temp:= HexToByte(CurChar);
            Data.Data[Idx]:= Temp;
            ChkSum:= ChkSum + Temp;
            CurChar:= Ptr(ord(CurChar) + 2)
          end
        end;
        Data.ChkSum:= HexToByte(CurChar);
        ChkSum:= (ChkSum xor $FF) + 1;
      end
      else begin
        WriteLn('Invalid line length. ', ReqLen:4, InLen:4);
        WasOk:= false
      end;
    end
    else begin
      WriteLn('Invalid character detected.')
    end
  end
  else begin
    Writeln('Invalid line format.')
  end;
  DecodeHexLine:= WasOk
end;

function DecodeFile( aFileName: TFileName;
                     FirstDataOffset: integer;
                     var BootRec: TBootRec;
                     var FirstAdr, LastAdr, LoadAdr:integer): boolean;

var
  F: Text;
  Line: TInputLine;
  Hex: THexLine;
  Valid, Working: boolean;
  ReqChkSum: byte;
  FCopyAdr, LCopyAdr: integer;

begin
  assign(F, aFileName);
  reset(F);
  Valid:= (IoResult = 0);
  if Valid then begin
    LastAdr:= -1;
    LoadAdr:= -1;
    Working:= true;
    while (not Eof(F)) and Valid and Working do begin
      ReadLn(F, Line);
      if (Line <> '') then begin
        Valid:= DecodeHexLine(Line, Hex, ReqChkSum);
        if Valid then begin
          if (Hex.ChkSum = ReqChkSum) then begin
            if (Hex.RecType = 0) then begin
              if (Hex.DataCount > 0) then begin
                if (LoadAdr = -1) then begin
                  LoadAdr:= Hex.Address;
                  if (FirstAdr = -1) then begin
                    FirstAdr:= Hex.Address
                  end
                end;
                LastAdr:= Hex.Address + Hex.DataCount - 1;
                {copy Data to Boot Record}
                FCopyAdr:= FirstDataOffset + (Hex.Address - FirstAdr);
                LCopyAdr:= (FCopyAdr + Hex.DataCount) -1;
                if (LCopyAdr < C_BootRecBytes) then begin
                  move(Hex.Data[0], BootRec[FCopyAdr], Hex.DataCount)
                end
                else begin
                  WriteLn('Image address is out of range', LCopyAdr:8);
                  Valid:= False
                end
              end
              else begin
                Working:= false
              end
            end
            else begin
              Valid:= false;
              Writeln(Line);
              WriteLn('Invalid record type.')
            end
          end
          else begin
            Valid:= false;
            Writeln(Line);
            WriteLn('Invalid check sum.')
          end
        end
      end
      else begin
        Working:= false
      end
    end;
    Close(F);
  end
  else begin
    writeln('Unable to open source file: ',aFileName)
  end;
  DecodeFile:= Valid
end;

procedure WriteHex(Data: integer);

var
  Ch, Idx: byte;
  St: string[4];

begin
  St:= '0000';
  for Idx:= 4 downto 1 do begin
    Ch:= (Data and 15);
    if (Ch > 9) then
      Ch:= Ch + ord('A') - 10
    else
      Ch:= Ch + ord('0');
    St[Idx]:= Char(Ch);
    Data:= Data shr 4
  end;
  write(St)
end;

function LoadBootImage(BootFile, CpmFile, BiosFile: TFileName;
                       var BootRec: TBootRec;
                       var FinalSize: integer): boolean;
var
  FirstAdr, LastAdr, LoadAdr:integer;
  LoaderSize: integer;

  procedure ShowAdrs;

  begin
    WriteHex(LoadAdr);
    write(' to ');
    WriteHex(LastAdr);
    writeln
  end;

begin
  LoadBootImage:= false;
  writeln('Building boot image...');
  fillchar(BootRec, sizeOf(TBootRec), 0);
  FirstAdr:= -1;
  LastAdr:= -1;
  Writeln('  Loading boot loader...');
  if DecodeFile(BootFile, 0, BootRec, FirstAdr, LastAdr, LoadAdr) then begin
    LoaderSize:= LastAdr;
    write('  Boot loader copied: ');
    ShowAdrs;
    if (FirstAdr = 0) and (LastAdr < C_MaxLoaderBytes) then begin
      FirstAdr:= -1;
      LastAdr:= -1;
      Writeln('  Loading CPM image...');
      if DecodeFile(CpmFile, C_MaxLoaderBytes, BootRec, FirstAdr, LastAdr, LoadAdr) then begin
        write('  CPM 2.2 image copied: ');
        ShowAdrs;
        writeln('  Loading BIOS...');
        if DecodeFile(BiosFile, C_MaxLoaderBytes, BootRec, FirstAdr, LastAdr, LoadADr) then begin
          write('  BIOS copied: ');
          ShowAdrs;
          writeln;
          writeln('Boot image created.');
          writeln('  Boot loader size: ', LoaderSize);
          write('  CPM CCP/BDOS address: ');
          WriteHex(FirstAdr);
          writeln;
          write('  BIOS address: ');
          WriteHex(LoadAdr);
          writeln;
          {calculate total image size}
          FinalSize:= (LastAdr - FirstAdr) + C_MaxLoaderBytes;
          LoadBootImage:= true
        end
        else begin
          writeln('Loader must start at address 0 and be 256 bytes or less in size.')
        end
      end
    end
  end
end;


function SaveBootImage(ImageFile: TFileName;
                       var BootRec: TBootRec;
                       ImageSize: integer): boolean;

var
  F: File;
  ImgRecs, Idx, WriteIdx: integer;

begin
  SaveBootImage:= false;
  ImgRecs:= (ImageSize + 127) div 128;
  assign(F, ImageFile);
  rewrite(F);
  if (IoResult = 0) then begin
    WriteIdx:= 0;
    for Idx:= 1 to ImgRecs do begin
      BlockWrite(F, BootRec[WriteIdx], 1);
      WriteIdx:= WriteIdx + 128
    end;
    SaveBootImage:= (IoResult = 0);
    Close(F)
  end
end;


procedure DspPrms;

var
  Cnt, Idx: byte;

begin
  Cnt:= ParamCount;
  for Idx:= 1 to Cnt do begin
    WriteLn(ParamStr(Idx))
  end
end;

var
  BootRec: TBootRec;
  TotalSize, TotalRecs: integer;
  BootFile, CpmFile, BiosFile: TFileName;

begin
  writeln;
  DspPrms;
  writeln;
  BootFile:= 'M80BOOT.HEX';
  CpmFile:= 'CPM22.HEX';
  BiosFile:= 'M80BIOS.HEX';
  if LoadBootImage(BootFile, CpmFile, BiosFile, BootRec, TotalSize) then begin
    writeln;
    writeln('Saving boot image...');
    Write('  Image size: ', TotalSize, ' bytes = ');
    Writeln((TotalSize + 127) div 128, ' records.');
    if SaveBootImage('M80BOOT.BIN', BootRec, TotalSize) then begin
      Writeln('Boot image saved.')
    end
    else begin
      Writeln('Unable to write boot image to disk.')
    end
  end
end.