'' Z80 control version 1.0
'' This object controls a Z80 CPU IC.
'' This objects acts as a boot loader and I/O for Z80.

CON
  _clkmode = xtal1 + pll16x             ' Set up the clock frequencies
  _xinfreq = 5_000_000


' Connections to Z80, D0...D7 on Z80 are connected to P0...P7 on Prop
  CPU_IOREQ_PIN = 8
  CPU_RESET_PIN = 9
  CPU_CLK_PIN   = 10
  CPU_RD_PIN    = 11
  CPU_WT_PIN    = 12
  CPU_EN_PIN    = 13
  CPU_ADRQ_PIN  = 14
  CPU_ADRCLK_PIN  = 15

' Z80 pin masks
  c_CpuIoReqMsk    =     1 << CPU_IOREQ_PIN                              ' Mask to read Z80 IO request line (/IOREQ)
  c_CpuResetMsk    =     1 << CPU_RESET_PIN                              ' Mask to set Z80 reset line (/RESET)
  c_CpuClkMsk      =     1 << CPU_CLK_PIN                                ' Mask to set Z80 clock line (CLK), also tied to LOAD/SHIFT pin on 74HC165
  c_CpuRdMsk       =     1 << CPU_RD_PIN                                 ' Mask to read/set Z80 data read line (/RD)
  c_CpuWtMsk       =     1 << CPU_WT_PIN                                 ' Mask to read/set Z80 data write line (/WT)
  c_CpuEnMsk       =     1 << CPU_EN_PIN                                 ' Mask to set enable pin for RD/WT line buffer (74LV367)
  c_CpuAdrDataMsk  =     1 << CPU_ADRQ_PIN                               ' Mask to read port address value from parallel to serial IC (74HC165)
  c_CpuAdrClkMsk   =     1 << CPU_ADRCLK_PIN                             ' Mask to set port address clock on parallel to serial IC (74HC165)

  c_CpuRdWtReqMsk  =     (c_CpuRdMsk | c_CpuWtMsk)                           
  c_BaseDirRegMsk  =    (c_CpuResetMsk |  c_CpuClkMsk | c_CpuEnMsk | c_CpuAdrClkMsk)

' COM port status bits
  c_ComAttached    = 1
  c_ComCanRead     = 2
  c_ComCanWrite    = 4
  c_ComTxEmpty     = 8
  
' COM port Clear fifo bits
  c_ComClrRead     = 2
  c_ComClrWrite    = 4

  c_Com1RxFifoSize = 256
  c_Com1TxFifoSize = 128

  c_Com2RxFifoSize = 256
  c_Com2TxFifoSize = 128

  c_Com3RxFifoSize = 256
  c_Com3TxFifoSize = 128

' Disk drive attributes
  c_DriveCanBoot = 1
  c_DriveWriteProtected = 2
  
' Disk status register,0-OK, 1-seek failed, 2-read failed, 4-write failed, 8-bad disk
  dcsb_seek_failed  = 1           ' set to 1 when a home disk request fails
  dcsb_read_failed  = 2           ' set to 1 when a read sector request fails
  dcsb_write_failed = 4           ' set to 1 when a write sector request fails 
  dcsb_bad_disk     = 8           ' set to 1 if disk index is out of range for mounted disk pack
  dcsb_Busy         = 16          ' set to 1 when disk controler is busy with last request

var
  long  Z80Cog
  'Six longs passed to COG at start up
  long  Z80MemLoadAddress  
  long  Z80LoadByteCount
  long  IoRequest
  'Disk controller sector buffer
  long  SectBufAdr             ' Address of sector data buffer
  long  BlockBufAdr            ' Address of sector buffer data index
  ' COM port fifo pointers (for all three com ports)
  long ComFifoDataAdr  
  
PUB Start(aBootLoaderAdr, aBootLoaderCount, aSectBufAdr, aBlockBufAdr, aComFifoDataAdr, aCpuClockFreq)


  if (Z80Cog <> 0)
    Stop
  BaseRdPortAdr:= @RdPorts
  BaseWrPortAdr:= @WrPorts  
  IoRequest:= 0
  Z80MemLoadAddress:= aBootLoaderAdr   
  Z80LoadByteCount:= aBootLoaderCount
  BlockBufAdr:= aBlockBufAdr
  SectBufAdr:= aSectBufAdr
  ComFifoDataAdr:= aComFifoDataAdr
  CpuClockFreq:= aCpuClockFreq 
  Z80Cog:= cognew(@T1, @Z80MemLoadAddress) + 1
  
pub Stop
  if Z80Cog <> 0
    cogstop(Z80Cog - 1)
    Z80Cog:= 0

pub WaitForIoReq: request
  if (Z80Cog == 0)
    request:= 0
  else
    {wait for CMD byte to not = 0}
    repeat until ((IoRequest & $F000_0000) <> 0)
    request:= IoRequest

pub RespondToRequest(Responce)
  IoRequest:= Responce
     
DAT

T1                      mov     T1, PAR
BootDataAdr             rdlong  BootDataAdr, T1
IoReqMemAdr             add     T1, #4
BootDataCnt             rdlong  BootDataCnt, T1
TmpHead                 add     T1, #4
TmpTail                 mov     IoReqMemAdr, T1
TmpAdr                  add     T1, #4
TmpNextHead             rdlong  DC_SectBufAdr, T1
IoAdr                   add     T1, #4
IoData                  rdlong  SD_BlockBufAdr, T1
                        'next location is address of 18 longs that hold com fifo pointers
T2                      add     T1, #4
T3                      rdlong  T1, T1
                        ' T1 = start of 42 longs
T4                      mov     T2, #42
Entry1                  rdlong  Com1RxHeadAdr, T1
FifoHeadAdr             add     Entry1, D0_INC
FifoTailAdr             add     T1, #4 
FifoDataAdr             djnz    T2, #Entry1
                        
                        'All parameters are read startup CPU
DC_SectRdAdr            mov     DC_SectRdAdr, SD_BlockBufAdr
FifoByteSize            jmp     #Z80_Run
'
'
' Reset the Z80 CPU and then stop the Z80 clock

Z80_Reset               mov     CTRB, #0                ' Disable counter B, no output on CPU_CLK_PIN               
                        mov     PHSB, #0
                        mov     FRQB, #0                ' Stop clock accumulation
                        mov     CTRB, #0                ' Disable counter A, no output on CPU_ADRCLK_PIN
                        ' Z80 clock is stopped, setup base pin directions
                        mov     OUTA, #0
                        mov     DIRA, BaseDirRegMsk
                        ' CPU reset is active, clock is low, and the RD/WT line buffer is enabled.
                        ' Run the clock for at least 8 to 10 CPU clock cycles before RESET is disabled
                        mov     T1, #10
                        mov     T2, CNT
                        add     T2, #32
                        'Set Z80 clock line high
'                        or      DIRA, CpuClkMsk
Z80_ResetLoop           or      OUTA, CpuClkMsk
                        waitcnt T2, #32
                        'Set Z80 clock line low
                        andn    OUTA, CpuClkMsk
                        waitcnt T2, #32
                        'See if we need more clock pulses
                        djnz    T1, #Z80_ResetLoop
                        'Release rest line, there is a 10k pullup on reset so just disable as output                        
                        or      OUTA, CpuResetMsk
                        andn    DIRA, CpuResetMsk
                        'Z80 has been reset, ready clock but do not start, then return
                        mov     CTRB, CpuClockMode      ' Set counter B to NCO mode, output on CPU_CLK_PIN
                        mov     FRQA, #0
                        mov     PHSA, #0
                        mov     CTRA, AdrClockMode      ' Set counter A to NCO mode, output on CPU_ADRCLK_PIN
Z80_Reset_ret           ret


' Reset the Z80 then load the data at the address BootDataAdr, with a byte count of BootDataCnt                        

Z80_Boot                call    #Z80_Reset
                        ' Disable CPU RD/WT lines
                        or      OUTA, CpuEnMsk
                        'Make Prop WT line an output, set it high, inactive
                        or      OUTA, CpuWtMsk
                        or      DIRA, CpuWtMsk
                        'Initialize our loop variables
                        mov     T3, BootDataAdr
                        mov     T4, BootDataCnt
                        ' Run Z80 until Z80 request first read, /RD line = 0
                        mov     FRQB, CpuClockFreq
Z80_BootNext            waitpne CpuRdMsk, CpuRdMsk      ' Wait for Z80 RD line to go low
                        mov     FRQB, #0                ' Stop Z80 clock accumulation
                        ' We have a read request, Z80 has a valid address and /MEMREQ signal.
                        ' Z80 is in read mode so data lines are inputs.
                        ' Save boot loader byte
                        rdbyte  T1, T3          'read next boot loader byte
                        add     T3, #1
                        'Place the data on the Z80/Ram data buss (P0..P7) 
                        andn    OUTA, #255
                        or      OUTA, T1
                        or      DIRA, #255      'Set data lines as outputs
                        'Data is output to ram, Z80 hold address and ram chip enable.
'                        nop                     'wait for ram to ready
                        'activate write line on ram
                        andn    OUTA, CpuWtMsk
'                        nop                     'wait for ram to ready
                        'Disable ram write signal
                        or      OUTA, CpuWtMsk
                        'Now feed the Z80 a NOP opcode ( nop = 0 )
                        andn    OUTA, #255
                        'Run Z80 until RD line is high
                        mov     FRQB, CpuClockFreq      'Run the Z80
                        waitpeq CpuRdMsk, CpuRdMsk      'Wait for RD line to go high
                        'Set the Data buss to inputs
                        andn    DIRA, #255      'Set data lines as inputs
                        djnz    T4, #Z80_BootNext
                        'Boot loader is copied to Z80 ram, rest CPU and we are ready to run
                        call    #Z80_Reset                 
Z80_Boot_ret            ret 
'
'
' These are the com port fifo support functions
'
{
FifoHeadAdr   long      0
FifoTailAdr   long      0
FifoDataAdr   long      0
FifoByteSize  long      0
}

' Read a byte from the fifo and return it in IoData
' If fifo is empty return last byte read

FifoRdNext              rdlong  TmpTail, FifoTailAdr
                        'we have at least one byte to read
                        mov     TmpAdr, FifoDataAdr
                        add     TmpAdr, TmpTail
                        rdbyte  IoData, TmpAdr                 'read the data from the fifo
                        'update the com RX fifo Tail if fifo not empty
                        rdlong  TmpHead, FifoHeadAdr
                        cmp     TmpHead, TmpTail          wz
        if_z            jmp     #FifoRdNextDone
                        add     TmpTail, #1
                        cmpsub  TmpTail, FifoByteSize     wc
                        wrlong  TmpTail, FifoTailAdr       'update fifo tail
FifoRdNextDone          jmp     #DoIoReadRun 

'
'
' Write a byte passed in IoData to the the fifo.
FifoWrNext              rdlong  TmpHead, FifoHeadAdr
                        mov     TmpNextHead, TmpHead
                        add     TmpNextHead, #1
                        rdlong  TmpTail, FifoTailAdr
                        cmpsub  TmpNextHead, FifoByteSize    wc
                        cmp     TmpNextHead, TmpTail         wz
                        'if_z no room to do write to fifo
        if_z            jmp     #DoIoWriteRun
                        'we can write the data bayte
                        mov     TmpAdr, FifoDataAdr
                        add     TmpAdr, TmpHead
                        wrbyte  IoData, TmpAdr                 'read the data from the fifo
                        wrlong  TmpNextHead, FifoHeadAdr  
                        jmp     #DoIoWriteRun
'
'
' Return read/write status for COM port in IoData
            
                        ' See if there is any data waiting in com RX fifo to be read
FifoDoStatus            rdlong  TmpHead, TmpHead
                        mov     IoData, #c_ComAttached     ' Indicate com port device is present
                        rdlong  TmpTail, TmpTail
                        cmp     TmpHead, TmpTail             wz              
                        muxnz   IoData, #c_ComCanRead     ' Set the can read status bit
                        ' See if there is any room in com TX fifo to write a byte
                        rdlong  TmpHead, FifoHeadAdr
                        mov     TmpNextHead, TmpHead
                        add     TmpNextHead, #1
                        cmpsub  TmpNextHead, FifoByteSize   
                        rdlong  TmpTail, FifoTailAdr
                        cmp     TmpNextHead, TmpTail         wz
                        muxnz   IoData, #c_ComCanWrite     ' Set can write status bit
                        ' see if all TX data sent
                        cmp     TmpHead, TmpTail             wz              
                        muxnz   IoData, #c_ComTxEmpty     ' Set the can read status bit
                        jmp     #DoIoReadRun 
'
'
' Clear read/write fifo for COM port, IoData data contains flags
            
                        ' See if we need to clear RX fifo
FifoDoClr               test    IoData, #c_ComClrRead      wz
              if_z      jmp     #:FifoClrTx
                        ' Clear Rx fifo
                        rdlong  TmpHead, TmpHead
                        wrlong  TmpHead, TmpTail
                        ' See if we need to clear TX fifo
:FifoClrTx              test    IoData, #c_ComClrWrite     wz
              if_z      jmp     #DoIoWriteRun
                        ' Clear Tx fifo
                        rdlong  TmpHead, FifoTailAdr
                        wrlong  TmpHead, FifoHeadAdr
                        jmp     #DoIoWriteRun
'
'
' Get read/write status for com1 port and return in IoData

Com1Status              mov     FifoHeadAdr, Com1TxHeadAdr
                        mov     FifoTailAdr, Com1TxTailAdr
                        mov     FifoByteSize, #c_Com1TxFifoSize
                        ' Next we need read status
                        mov     TmpHead, Com1RxHeadAdr
                        mov     TmpTail, Com1RxTailAdr
                        ' Calc status bits
                        jmp     #FifoDoStatus
'                        
'
' Get read/write status for com2 port and return in IoData
Com2Status              mov     FifoHeadAdr, Com2TxHeadAdr
                        mov     FifoTailAdr, Com2TxTailAdr
                        mov     FifoByteSize, #c_Com2TxFifoSize
                        ' Next we need read status
                        mov     TmpHead, Com2RxHeadAdr
                        mov     TmpTail, Com2RxTailAdr
                        ' Calc status bits
                        jmp     #FifoDoStatus
'
'                        
' Get read/write status for com2 port and return in IoData
Com3Status              mov     FifoHeadAdr, Com3TxHeadAdr
                        mov     FifoTailAdr, Com3TxTailAdr
                        mov     FifoByteSize, #c_Com3TxFifoSize
                        ' Next we need read status
                        mov     TmpHead, Com3RxHeadAdr
                        mov     TmpTail, Com3RxTailAdr
                        ' Calc status bits
                        jmp     #FifoDoStatus
'                        
'
' Write a byte passed in IoData to the COM1 TX fifo.
Com1WrNext              mov     FifoHeadAdr, Com1TxHeadAdr
                        mov     FifoTailAdr, Com1TxTailAdr
                        mov     FifoDataAdr, Com1TxBufAdr
                        mov     FifoByteSize, #c_Com1TxFifoSize
                        jmp     #FifoWrNext
'

' Write a byte passed in IoData to the COM2 TX fifo.
Com2WrNext              mov     FifoHeadAdr, Com2TxHeadAdr
                        mov     FifoTailAdr, Com2TxTailAdr
                        mov     FifoDataAdr, Com2TxBufAdr
                        mov     FifoByteSize, #c_Com2TxFifoSize
                        jmp     #FifoWrNext
'

' Write a byte passed in IoData to the COM3 TX fifo.
Com3WrNext              mov     FifoHeadAdr, Com3TxHeadAdr
                        mov     FifoTailAdr, Com3TxTailAdr
                        mov     FifoDataAdr, Com3TxBufAdr
                        mov     FifoByteSize, #c_Com3TxFifoSize
                        jmp     #FifoWrNext
'
'
' Read a byte from the COM1 RX fifo. Byte is returned in IoData
Com1RdNext              mov     FifoHeadAdr, Com1RxHeadAdr
                        mov     FifoTailAdr, Com1RxTailAdr
                        mov     FifoDataAdr, Com1RxBufAdr
                        mov     FifoByteSize, #c_Com1RxFifoSize
                        jmp     #FifoRdNext
'
'
' Read a byte from the COM2 RX fifo. Byte is returned in IoData
Com2RdNext              mov     FifoHeadAdr, Com2RxHeadAdr
                        mov     FifoTailAdr, Com2RxTailAdr
                        mov     FifoDataAdr, Com2RxBufAdr
                        mov     FifoByteSize, #c_Com2RxFifoSize
                        jmp     #FifoRdNext
'
'
' Read a byte from the COM3 RX fifo. Byte is returned in IoData
Com3RdNext              mov     FifoHeadAdr, Com3RxHeadAdr
                        mov     FifoTailAdr, Com3RxTailAdr
                        mov     FifoDataAdr, Com3RxBufAdr
                        mov     FifoByteSize, #c_Com3RxFifoSize
                        jmp     #FifoRdNext
'
'
' Clear the selected Rx/Tx fifo for COM1
Com1ClrFifo             mov     FifoHeadAdr, Com1TxHeadAdr
                        mov     FifoTailAdr, Com1TxTailAdr
                        ' Next we need read status
                        mov     TmpHead, Com1RxHeadAdr
                        mov     TmpTail, Com1RxTailAdr
                        ' Calc status bits
                        jmp     #FifoDoClr
'
'
' Clear the selected Rx/Tx fifo for COM2
Com2ClrFifo             mov     FifoHeadAdr, Com2TxHeadAdr
                        mov     FifoTailAdr, Com2TxTailAdr
                        ' Next we need read status
                        mov     TmpHead, Com2RxHeadAdr
                        mov     TmpTail, Com2RxTailAdr
                        ' Calc status bits
                        jmp     #FifoDoClr
'
' Clear the selected Rx/Tx fifo for COM3
Com3ClrFifo             mov     FifoHeadAdr, Com3TxHeadAdr
                        mov     FifoTailAdr, Com3TxTailAdr
                        ' Next we need read status
                        mov     TmpHead, Com3RxHeadAdr
                        mov     TmpTail, Com3RxTailAdr
                        ' Calc status bits
                        jmp     #FifoDoClr
'
'
' SD card block read/wrote functions
'
' Transfer a block (512 butes) to/from the SD card, T1 = block index to transfer.
' The source of the opcode at SD_RequestIOCmd must be set to "r" or "w".  
' On exit T1 = 0 if operation ok, <> 0 means an error.
SD_RequestIO            mov     T2, DC_SpiCmdAdr
                        add     T2, #4
                        wrlong  T1, T2                  ' Set block index for SD command
                        add     T2, #4
                        wrlong  SD_BlockBufAdr, T2      ' Set the SD block buffer address for SD command 
                        ' Now issue the command
SD_RequestIOCmd         mov     T2, #"r"
                        wrlong  T2, DC_SpiCmdAdr
                        ' Wait for command to complete
SD_RequestIOWait        rdlong  T1, DC_SpiCmdAdr
                        cmp     T1, T2          wz
              if_z      jmp     #SD_RequestIOWait
                        ' If T1 = 0 task completed, else return and report error in T1
SD_RequestIO_ret        ret        
'
'
DesiredBlockNum         long    0

'
GetSelectedDiskIdx      mov     T1, DC_DiskSelect
                        and     T1, #3
                        add     T1, #DC_Disk0Index
                        movs    GetSelectedDiskIdxRd, T1    
                        nop
GetSelectedDiskIdxRd    mov     T1, (DC_Disk0Index)                        
GetSelectedDiskIdx_ret  ret
'
'
' Read the attributes register for the selected drive.
' Attributes are returned in T1.
GetSelDiskAttribs       mov     T1, DC_DiskSelect
                        and     T1, #3
                        add     T1, #DC_Disk0Attribs
                        movs    GetSelDiskAttribsRd, T1    
                        nop
GetSelDiskAttribsRd     mov     T1, (DC_Disk0Attribs)                        
GetSelDiskAttribs_ret   ret
'
'
' Calculate the SD card block index for the CPM sector address and return in T1.
                        ' Fist calculate starting block index of selecetd disk
DC_CalcSdBlockNum       call    #GetSelectedDiskIdx                       ' Get selected disk index
                        mov     DesiredBlockNum, T1                   
                        shl     DesiredBlockNum, DC_DiskToBlocksShift     ' Multiply disk index by SD card blocks per disk
                        add     DesiredBlockNum, DC_FirstDiskBlock        ' Add offset to start on disk area
                        ' DesiredBlockNum = SD block index to start of selecetd disk
                        ' Now calculate block offset of selected sedctor
                        mov     T2, DC_CpmTrackNumH          ' Get selected disk track index
                        shl     T2, #8
                        or      T2, DC_CpmTrackNumL
                        shl     T2, DC_TrackToSectShift      ' Multiply track index by cpm sectors per track
                        add     T2, DC_CpmSectorNumL         ' Add selected cpm track sector number
                        ' Convert std. cpm sector count to SD card block count
                        shr     T2, #2          ' Divide by 4 (512 / 128)
                        ' Now add disk start block in DesiredBlockNum to disk sector/track block offset in T2
                        add     DesiredBlockNum, T2
DC_CalcSdBlockNum_ret   ret
'                                
'
' Flush the SD buffer if it is dirty, T1 = 0 if Ok else T1 <> 0
DC_FlushBuffer          mov     T1, #0                                          ' Indicate no error
                        tjz     SD_BufferDirty, #DC_FlushBuffer_ret
                        mov     T1, SD_BlockIndex
                        ' Write a block (512 butes) to the SD card, T1 = block index to write
                        movs    SD_RequestIOCmd, #"w"
                        call    #SD_RequestIO
                        ' See if write went OK
                        tjnz    T1, DC_FlushBuffer_ret
                        mov     SD_BufferDirty, #0      ' Indicate that the buffer is the same as data on SD xard
DC_FlushBuffer_ret      ret
'
'
'
' Read a cpm sector from the disk, T1 = 0 if OK, T1 <> 0 failed read
DC_ReadSector           call    #GetSelectedDiskIdx                       ' Get selected disk index
                        cmp     T1, DC_DiskPackDiskCount  wc
              if_nc     jmp     DC_ReadSector_ret
                        ' Disk index is in range, calculate SD card block we need to read from     
                        call    #DC_CalcSdBlockNum
                        ' See if block is in SD block buffer
                        cmp     DesiredBlockNum, SD_BlockIndex       wz
              if_z      jmp     #DC_ReadSectorCopy
                        ' Sector not in buffer, if buffer is dirty flush it
                        call    #DC_FlushBuffer
                        ' Read sector from buffer
                        tjnz    T1, #DC_ReadSector_ret
                        ' Buffer flushed, now read buffer we need
                        mov     T1, DesiredBlockNum
                        ' Read a block (512 butes) from the SD card, T1 = block index to read
                        movs    SD_RequestIOCmd, #"r"
                        call    #SD_RequestIO
                        ' See if read OK
                        tjnz    T1, #DC_ReadSector_ret
                        ' Update block buffer index
                        mov     SD_BlockIndex, DesiredBlockNum
                        mov     SD_BufferDirty, #0      ' Indicate that the buffer is the same as data on SD card
                        ' We have have the block index i the buffer, copy sector we need
DC_ReadSectorCopy       mov     DC_SecDataIndex, #0
                        ' Calculate offset into SD block buffer
                        mov     T1, DC_CpmSectorNumL
                        and     T1, #3          ' get cpm sector index in block buffer (0,,3(
                        shl     T1, #7          ' Mult by 128 to get byte offset in SD block buffer
                        add     T1, SD_BlockBufAdr
                        ' T1 = memory address of first byte to copy to sector buffer
                        mov     DC_SectRdAdr, T1
                        ' Indicate no error
                        mov     T1, #0
DC_ReadSector_ret       ret
'                                      
'
' Write a cpm sector to the disk, T1 = 0 if OK, T1 <> 0 failed write
DC_WriteSector          call    #GetSelectedDiskIdx                       ' Get selected disk index
                        cmp     T1, DC_DiskPackDiskCount  wc
              if_nc     jmp     DC_WriteSector_ret
                        ' Disk index is in range, see if drive is write protected     
                        call    #GetSelDiskAttribs
                        and     T1, #c_DriveWriteProtected
                        tjnz    T1, #DC_WriteSector_ret 
                        ' Disk index is in range and not write protected, calculate SD card block we need to write to     
                        call    #DC_CalcSdBlockNum
                        ' See if block is in SD block buffer
                        cmp     DesiredBlockNum, SD_BlockIndex       wz
              if_z      jmp     #DC_WriteSectorCopy
                        ' Sector not in buffer, if buffer is dirty flush it
                        call    #DC_FlushBuffer
                        tjnz    T1, #DC_WriteSector_ret
                        ' Buffer flushed, now write buffer we need
                        ' Read desired block from SD card to buffer
                        mov     T1, DesiredBlockNum
                        ' Read a block (512 butes) from the SD card, T1 = block index to read
                        movs    SD_RequestIOCmd, #"r"
                        call    #SD_RequestIO
                        ' See if read OK
                        tjnz    T1, #DC_WriteSector_ret
                        ' Update block buffer
                        mov     SD_BlockIndex, DesiredBlockNum
                        ' We have have the block index ib the block buffer, copy sector we need
DC_WriteSectorCopy      mov     DC_SecDataIndex, #0
                        ' Calculate offset into SD block buffer
                        mov     T1, DC_CpmSectorNumL
                        and     T1, #3          ' get cpm sector index in block buffer (0,,3(
                        shl     T1, #7          ' Mult by 128 to get byte offset in SD block buffer
                        add     T1, SD_BlockBufAdr
                        ' T1 = memory address of first byte to copy to sector buffer
                        mov     T2, DC_SectBufAdr
                        ' The sector buffer and SD block buffer both start at a long aligned address
                        ' and the byte offset is alway a mult of 4 so can copy as longs
                        mov     T3, #32         ' Number of longs yo transfer (128/4)
DC_WriteSectorRpt       rdlong  T4, T2          ' Read next long from sector buffer
                        add     T2, #4          ' Update next read memory address
                        wrlong  T4, T1          ' Copy the long value to the block buffer
                        add     T1, #4          ' Update next write memory address
                        djnz    T3, #DC_WriteSectorRpt
                        ' Indicate that buffer is dirty
                        mov     SD_BufferDirty, #1      ' Indicate that the buffer is not the same as data on SD card     
                        ' Indicate no error
                        mov     T1, #0
DC_WriteSector_ret      ret
'                                      
''
' Perform a disk controller command.
' The command index is in the cariable IoData
' The disk status is updated with results.
' Disk command register,  0-reset controller, 1-home, 2-read, 3-write
DC_DoCommand            mov     DC_LastStatus, #0
                        cmp     IoData, #4      wc
              if_nc     jmp     #DC_DoCommand_ret
                        ' command in range                                         
                        mov     T1, IoData
                        shl     T1, #2
                        add     T1, #DC_DoCommandReset
                        jmp     T1
                        ' Reset disk controller
DC_DoCommandReset       call    #DC_FlushBuffer
                        jmp     #DC_DoCommand_ret
                        nop
                        nop
                        ' Home selected disk
DC_DoCommandHome        call    #DC_FlushBuffer
                        cmp     T1, #0          wz
              if_nz     or      DC_LastStatus, #dcsb_bad_disk                        
                        jmp     #DC_DoCommand_ret
                        ' Read a cpm sector from disk
DC_DoCommandRead        call    #DC_ReadSector
                        cmp     T1, #0          wz
              if_nz     or      DC_LastStatus, #dcsb_read_failed                        
                        jmp     #DC_DoCommand_ret
                        ' Write a cpm disk sector
DC_DoCommandWrite       call    #DC_WriteSector
                        cmp     T1, #0          wz
              if_nz     or      DC_LastStatus, #dcsb_write_failed                        
DC_DoCommand_ret        jmp     #DoIoWriteRun                                       
'
'
' Read the memory location for a disk register
DC_IoRdRegister         and     IoAdr, #$0F
                        add     IoAdr, #DC_LastStatus
                        movs    DC_IoRdRegisterWr, IoAdr
                        nop
DC_IoRdRegisterWr       mov     IoData, (DC_LastStatus)
                        jmp     #DoIoReadRun            ' Return to caller
'
' Read disk controller next data port, byte returned in IoData
DC_IoRdDataRegister     mov     T1, DC_SecDataIndex     ' Get current sector data index value
                        add     T1, DC_SectRdAdr        ' T1 = memory address of sector byte in SD buffer
                        rdbyte  IoData, T1              ' IoData = data byte from sector buffer
                        'advance sector byte index
                        add     DC_SecDataIndex, #1
                        and     DC_SecDataIndex, #127
                        jmp     #DoIoReadRun
'
'
' Perform a disk controller port write. Port address range ($10..$1F)
' Register $16 is the sector buffer read data register.
' Register $10 is command register
' Data is in variable IoData

' Perform a disk controler write to register, data byte is in IoData, Port address is in IoAdr
DC_IoWrRegister         and     IoAdr, #$0F
                        add     IoAdr, #DC_LastStatus
                        movd    DC_IoWrRegisterWr, IoAdr
                        nop
DC_IoWrRegisterWr       mov     (DC_LastStatus), IoData
                        jmp     #DoIoWriteRun     ' Return to caller
'
'
' Write data to disk controller sector data port, byte to be written is in IoData
DC_IoWrDataRegister     mov     T1, DC_SecDataIndex                             ' Get current sector data index value
                        add     T1, DC_SectBufAdr                               ' T1 = memory address of sector buffer byte
                        wrbyte  IoData, T1                                      ' T2 = data byte from secor buffer
                        'advance sector byte index
                        add     DC_SecDataIndex, #1
                        and     DC_SecDataIndex, #127
                        jmp     #DoIoWriteRun                             ' Return to caller

'
' SD FAT file stream read next byte from sector buffer
FatRdDataRegister       mov     T1, DC_SecDataIndex     ' Get current sector data index value
                        add     T1, DC_SectBufAdr        ' T1 = memory address of sector byte in SD buffer
                        rdbyte  IoData, T1              ' IoData = data byte from sector buffer
                        'advance sector byte index
                        add     DC_SecDataIndex, #1
                        and     DC_SecDataIndex, #127
                        jmp     #DoIoReadRun
'

'
'
' Z80 request a read from an I/O port
'
                      ' See if we have a port this COG handles
DoIoRead                cmp     IoAdr, #$20     wc
              if_nc     jmp     #DoDefaultIoRead
                        ' Port is in 0..$1F range
                        mov     T1, IoAdr
                        shl     T1,#1                ' Port jump table address values are word size
                        add     T1, BaseRdPortAdr
                        rdword  T1, T1
                        tjnz    T1, T1 
                        ' Handle any port read not handled by this COG
                        ' Port is in IoAdr, IoData = data byte value
                        ' Not a com or disk controller register, do default action                        
DoDefaultIoRead         mov     T1, #1          ' Indicate a port read request
                        shl     T1, #22
                        or      T1, IoAdr
                        shl     T1, #8
                        ' Perform the request
                        wrlong  T1, IoReqMemAdr
                        'Wait for request to be completed
DoIoReadWait            rdlong  IoData, IoReqMemAdr
                        test    IoData, CommandMask         wz  
              if_nz     jmp     #DoIoReadWait
                        'Command completed
                        '
                        'Place the data on the Z80 data buss (P0..P7) 
DoIoReadRun             and     IoData, #255
                        andn    OUTA, #255
                        or      OUTA, IoData
                        or      DIRA, #255      'Set data lines as outputs
                        'Data is output to Z80.
                        'Run Z80 until RD line is high
                        jmp     #Z80_RunNext
'
' Do a default IO request to Handle any port write not handled by this COG
' Port is in IoAdr, IoData = data byte value
DefaultIoWrite          mov     T1, #2          'Indicate a write request
                        shl     T1, #22
                        or      T1, IoAdr
                        shl     T1, #8
                        or      T1, IoData
                        'Perform the request
                        wrlong  T1, IoReqMemAdr
                        'Wait for request to be completed
DoIoWriteWait           rdlong  T1, IoReqMemAdr
                        test    T1, CommandMask         wz  
              if_nz     jmp     #DoIoWriteWait
DefaultIoWrite_ret      ret

'
'Z80 request a write to an I/O port
DoIoWrite               mov     IoData, INA
                        and     IoData, #255
                        ' Port is in IoAdr and data byte is in IoData
                        cmp     IoAdr, #$20     wc
              if_nc     jmp     #DoDefaultIoWrite
                        ' Port is in 0..$1F range
                        mov     T1, IoAdr
                        shl     T1, #1               ' Port jump table address values are word size
                        add     T1, BaseWrPortAdr
                        rdword  T1, T1
                        tjnz    T1, T1 
                        ' If not just do a default IO request
                        ' Handle any port write not handled by this COG
                        ' Port is in IoAdr, IoData = data byte value
DoDefaultIoWrite        call    #DefaultIoWrite
                        ' Run Z80 until WT line is high
DoIoWriteRun            jmp     #Z80_RunNext
'
'
'
' Set the clock frequency used to drive the Z80 master clock
' IoData byte value is multiplied 100,000 by the default spin port action with same address.
' The spin code calculates the FRQB value to use and returns it.
DoSetZ80Freq            mov     IoData, INA
                        and     IoData, #255
                        ' Port is in IoAdr, IoData = data byte value
                        call    #DefaultIoWrite
                        ' FRQB value is in lower 28 bits of responce in T1
                        mov     CpuClockFreq, T1
                        shl     CpuClockFreq, #2
                        jmp     #Z80_RunNext
'              
' After the Z80 boot loader is completed run the Z80 CPU and hook IO requests
Z80_Run                 call    #Z80_Boot
                        ' Z80 boot loader is in Z80 ram ready to run
Z80_RunNext             mov     FRQB, CpuClockFreq      'Run the Z80
                        ' Wait for no read or write request from CPU
Z80_RunContunue         waitpeq CpuRdWtReqMsk, CpuRdWtReqMsk
                        andn    DIRA, #255              'Set data lines as inputs
                        ' Wait for a read or write request from CPU
                        waitpne CpuRdWtReqMsk, CpuRdWtReqMsk
                        ' If the IO/REQ line is low we have a read/write IO request from Z80
                        test    CpuIoReqMsk, INA        wz    
                        'If no io request wait for RD/WT to go high
              if_nz     jmp     #Z80_RunContunue
                        mov     FRQB, #0                ' Stop Z80 clock accumulation
                        ' We have a read/write request
                        ' Read the 8 bit I/O address from the Z80 using 74HC165 shift register.
                        ' This routine uses the A counter to clock the serial data transfer.
                        ' At a clock speed of 80Mhz data is clocked at 10Mhz (divide by 8).
                        '
                        ' Set address shift register (74HC165) load pin high, same pin as Z80 clock
                        movi    PHSB,  #%111_111111
                        '                                 
                        ' Use counter A to clock in the low 8 bits of Z80 address buss
                        ' Set up shift register clock, Bit 31 = 0 so adr clock is low
                        movi    PHSA, #%011_000000
                        movi    FRQA, #%001_000000
                        ' First data bit is ready to read, address is lsb first
                        test    CpuAdrDataMsk, INA      wc
                        ' Add data bit value to result
                        rcr     IoAdr, #1
                        ' Read next bit....
                        test    CpuAdrDataMsk, INA      wc
                        rcr     IoAdr, #1
                        test    CpuAdrDataMsk, INA      wc
                        rcr     IoAdr, #1
                        test    CpuAdrDataMsk, INA      wc
                        rcr     IoAdr, #1
                        test    CpuAdrDataMsk, INA      wc
                        rcr     IoAdr, #1
                        test    CpuAdrDataMsk, INA      wc
                        rcr     IoAdr, #1
                        test    CpuAdrDataMsk, INA      wc
                        rcr     IoAdr, #1
                        test    CpuAdrDataMsk, INA      wc
                        'Stop clock last bit has already been read
                        mov     FRQA, #0
                        'Save last bit of address byte (msb) to result
                        rcr     IoAdr, #1
                        'Move address byte to LSB of result
                        shr     IoAdr, #24
                        '
                        ' Finally ready to perform IO request    
                        ' Dispatch read or write, IO address in IoAdr
                        test    CpuRdMsk, INA        wz    
              if_nz     jmp    #DoIoWrite
                        'We have a port read request from Z80
                        jmp     #DoIoRead
'
'

' Z80 pin masks
  CpuIoReqMsk           long    c_CpuIoReqMsk   ' Mask to read Z80 IO request line (/IOREQ)
  CpuResetMsk           long    c_CpuResetMsk   ' Mask to set Z80 reset line (/RESET)
  CpuClkMsk             long    c_CpuClkMsk     ' Mask to set Z80 clock line (CLK), also tied to LOAD/SHIFT pin on 74HC165
  CpuRdMsk              long    c_CpuRdMsk      ' Mask to read/set Z80 data read line (/RD)
  CpuWtMsk              long    c_CpuWtMsk      ' Mask to read/set Z80 data write line (/WT)
  CpuEnMsk              long    c_CpuEnMsk      ' Mask to set enable pin for RD/WT line buffer (74LV367)
  CpuAdrDataMsk         long    c_CpuAdrDataMsk ' Mask to read port address value from parallel to serial IC (74HC165)
  CpuAdrClkMsk          long    c_CpuAdrClkMsk  ' Mask to set port address clock on parallel to serial IC (74HC165)
'  
' Base I/O pin direction register
  BaseRdPortAdr         long    0                       ' Read port jump table address in global memory 
  BaseWrPortAdr         long    0                       ' Write port jump table address in global memory 
  BaseDirRegMsk         long    c_BaseDirRegMsk
  CpuClockMode          long    (%00100 << 26) | (CPU_CLK_PIN << 0)             ' NCO, 50% duty cycle, used to clock Z80 CPU
  AdrClockMode          long    (%00100 << 26) | (CPU_ADRCLK_PIN << 0)          ' NCO, 50% duty cycle, used to clock address register (74HC165)
  CpuClockFreq          long    $2000_0000                                      ' Set to output 10Mhz on CPU_CLK_PIN (80Mhz / 8)
  CpuRdWtReqMsk         long    c_CpuRdWtReqMsk                           
  CommandMask           long    $C000_0000                                      ' Used to seperate command index from long buffer.
  D0_INC                long    %00000010_00000000

  ' SD block buffer variables
  SD_BlockIndex         long    0                       ' SD card block index loaded into SD block buffer
  SD_BufferDirty        long    0                       ' Indicates if buffer modified since read from SD

' All remaining variable are passed at startup

  'COM1 RX Fifo Variable Addresses
  Com1RxHeadAdr         long    0
  Com1RxTailAdr         long    0
  Com1RxBufAdr          long    0
  'COM1 TX Fifo Variable Addresses
  Com1TxHeadAdr         long    0
  Com1TxTailAdr         long    0
  Com1TxBufAdr          long    0
                                
  'COM2 RX Fifo Variable Addresses
  Com2RxHeadAdr         long    0
  Com2RxTailAdr         long    0
  Com2RxBufAdr          long    0
  'COM2 TX Fifo Variable Addresses
  Com2TxHeadAdr         long    0
  Com2TxTailAdr         long    0
  Com2TxBufAdr          long    0
                                
  'COM2 RX Fifo Variable Addresses
  Com3RxHeadAdr         long    0
  Com3RxTailAdr         long    0
  Com3RxBufAdr          long    0
  'COM3 TX Fifo Variable Addresses
  Com3TxHeadAdr         long    0
  Com3TxTailAdr         long    0
  Com3TxBufAdr          long    0                                

  ' SD block buffer variables                           
  DC_SpiCmdAdr          long    0                       ' Address of long used to pass commands to SD spi interface COG
  SD_BlockBufAdr        long    0                       ' Address of disk controller SD card block buffer

  ' Disk controller variables                           
  DC_SectBufAdr         res     1                       ' Address of disk controller sector data buffer
  DC_FirstDiskBlock     res     1                       ' SD block index of first block of disk area in disk pack file
  DC_DiskToBlocksShift  res     1                       ' Used to convert a disk number to block number offset
  DC_TrackToSectShift   res     1                       ' Used to convert a CPM track index to a std. sector index
  DC_DiskPackDiskCount  res     1                       ' Number of CPM disks in the disk pack
  
  ' Disk controler registers, these must be in order
  DC_LastStatus         res     1                       ' holds status from last disk controller command 
  DC_DiskSelect         res     1                       ' Selected disk drive (0..3), actual disk index stored in SC_Disk_Index variab;es 
  DC_CpmSectorNumL      res     1
  DC_CpmSectorNumH      res     1
  DC_CpmTrackNumL       res     1
  DC_CpmTrackNumH       res     1
  DC_SecData            res     1                       ' dumy register never used but must be here
  DC_SecDataIndex       res     1                       ' Disk drive controller sector buffer data index
  DC_Disk0Index         res     1                       ' Drive select 0 disk pack disk index  
  DC_Disk1Index         res     1                       ' Drive select 1 disk pack disk index  
  DC_Disk2Index         res     1                       ' Drive select 2 disk pack disk index  
  DC_Disk3Index         res     1                       ' Drive select 3 disk pack disk index  
  DC_Disk0Attribs       res     1                       ' Drive 0 disk attributes: b0 = bootable, b1 = write protected  
  DC_Disk1Attribs       res     1                       ' Drive 1 disk attributes: b0 = bootable, b1 = write protected  
  DC_Disk2Attribs       res     1                       ' Drive 2 disk attributes: b0 = bootable, b1 = write protected
  DC_Disk3Attribs       res     1                       ' Drive 3 disk attributes: b0 = bootable, b1 = write protected

{  
' Sytem parameters and variables
  BootDataAdr           res     1                       ' Global memory addres of Z80 boot loader image.
  BootDataCnt           res     1                       ' Byte count of Z80 boot loader image. 
  IoReqMemAdr           res     1                       ' Holds address of IO/request command buffer
}
  {
  ' Variables for a Z80 IO operation
  IoAdr                 res     1                                               ' Holds current IO request port address (0..255)
  IoData                res     1                                               ' Holds current port data value
  
' Working variables
  T1                    res     1       
  T2                    res     1       
  T3                    res     1       
  T4                    res     1       
  TmpHead               res     1       
  TmpTail               res     1       
  TmpAdr                res     1       
  TmpNextHead           res     1       
  }

padding long  0 [4]
'
' Port Read vector table, used for ports 0..31
'
RdPorts
  ' First group are com port Read and Status registers
  word        (@Com1RdNext - @T1) >> 2          ' port 0,  $00
  word        (@Com1Status - @T1) >> 2          ' port 1,  $01
  word        (@Com2RdNext - @T1) >> 2          ' port 2,  $02
  word        (@Com2Status - @T1) >> 2          ' port 3,  $03
  word        (@Com3RdNext - @T1) >> 2          ' port 4,  $04
  word        (@Com3Status - @T1) >> 2          ' port 5,  $05
  ' Reserved Registers, currently unused group of registers
  word        (@DoIoReadRun - @T1) >> 2         ' port 6,  $06
  word        (@DoIoReadRun - @T1) >> 2         ' port 7,  $07
  word        (@DoDefaultIoRead - @T1) >> 2     ' port 8,  $08
  word        (@DoIoReadRun - @T1) >> 2         ' port 9,  $09
  word        (@DoIoReadRun - @T1) >> 2         ' port 10, $0A
  word        (@DoIoReadRun - @T1) >> 2         ' port 11, $0B
  word        (@DoIoReadRun - @T1) >> 2         ' port 12, $0C
  word        (@DoIoReadRun - @T1) >> 2         ' port 13, $0D
  word        (@DoIoReadRun - @T1) >> 2         ' port 14, $0E
  word        (@FatRdDataRegister - @T1) >> 2   ' port 15, $0F
  ' Disk controller registers
  word        (@DC_IoRdRegister - @T1) >> 2     ' port 16, $10
  word        (@DC_IoRdRegister - @T1) >> 2     ' port 17, $11
  word        (@DC_IoRdRegister - @T1) >> 2     ' port 18, $12
  word        (@DC_IoRdRegister - @T1) >> 2     ' port 19, $13
  word        (@DC_IoRdRegister - @T1) >> 2     ' port 20, $14
  word        (@DC_IoRdRegister - @T1) >> 2     ' port 21, $15
  word        (@DC_IoRdDataRegister - @T1) >> 2 ' port 22, $16
  word        (@DC_IoRdRegister - @T1) >> 2     ' port 23, $17
  word        (@DC_IoRdRegister - @T1) >> 2     ' port 24, $18
  word        (@DC_IoRdRegister - @T1) >> 2     ' port 25, $19
  word        (@DC_IoRdRegister - @T1) >> 2     ' port 26, $1A
  word        (@DC_IoRdRegister - @T1) >> 2     ' port 27, $1B
  word        (@DC_IoRdRegister - @T1) >> 2     ' port 28, $1C
  word        (@DC_IoRdRegister - @T1) >> 2     ' port 29, $1D
  word        (@DC_IoRdRegister - @T1) >> 2     ' port 30, $1E
  word        (@DC_IoRdRegister - @T1) >> 2     ' port 31, $1F
'
' Port Write vector table, used for ports 0..31
'
WrPorts
  ' First group are com port Write data registers. Status registers are Read Only
  word        (@Com1WrNext - @T1) >> 2          ' port 0,  $00
  word        (@Com1ClrFifo - @T1) >> 2         ' port 1,  $01
  word        (@Com2WrNext - @T1) >> 2          ' port 2,  $02
  word        (@Com2ClrFifo - @T1) >> 2         ' port 3,  $03
  word        (@Com3WrNext - @T1) >> 2          ' port 4,  $04
  word        (@Com1ClrFifo - @T1) >> 2         ' port 5,  $05
  ' Reserved Registers, currently unused group of registers
  word        (@DoIoWriteRun - @T1) >> 2        ' port 6,  $06
  word        (@DoIoWriteRun - @T1) >> 2        ' port 7,  $07
  word        (@DoSetZ80Freq - @T1) >> 2        ' port 8,  $08
  word        (@DoIoWriteRun - @T1) >> 2        ' port 9,  $09
  word        (@DoIoWriteRun - @T1) >> 2        ' port 10, $0A
  word        (@DoIoWriteRun - @T1) >> 2        ' port 11, $0B
  word        (@DoIoWriteRun - @T1) >> 2        ' port 12, $0C
  word        (@DoIoWriteRun - @T1) >> 2        ' port 13, $0D
  word        (@DoIoWriteRun - @T1) >> 2        ' port 14, $0E
  word        (@DoIoWriteRun - @T1) >> 2        ' port 15, $0F
  ' Disk controller registers
  word        (@DC_DoCommand - @T1) >> 2        ' port 16, $10
  word        (@DC_IoWrRegister - @T1) >> 2     ' port 17, $11
  word        (@DC_IoWrRegister - @T1) >> 2     ' port 18, $12
  word        (@DC_IoWrRegister - @T1) >> 2     ' port 19, $13
  word        (@DC_IoWrRegister - @T1) >> 2     ' port 20, $14
  word        (@DC_IoWrRegister - @T1) >> 2     ' port 21, $15
  word        (@DC_IoWrDataRegister - @T1) >> 2 ' port 22, $16
  word        (@DC_IoWrRegister - @T1) >> 2     ' port 23, $17
  word        (@DC_IoWrRegister - @T1) >> 2     ' port 24, $18
  word        (@DC_IoWrRegister - @T1) >> 2     ' port 25, $19
  word        (@DC_IoWrRegister - @T1) >> 2     ' port 26, $1A
  word        (@DC_IoWrRegister - @T1) >> 2     ' port 27, $1B
  word        (@DC_IoWrRegister - @T1) >> 2     ' port 28, $1C
  word        (@DC_IoWrRegister - @T1) >> 2     ' port 29, $1D
  word        (@DC_IoWrRegister - @T1) >> 2     ' port 30, $1E
  word        (@DC_IoWrRegister - @T1) >> 2     ' port 31, $1F
'
'

         